/*
    I2C Master/Slave library for the Raspberry Pi Pico RP2040

    Copyright (c) 2021 Earle F. Philhower, III <earlephilhower@yahoo.com>

    Based off of TWI/I2C library for Arduino Zero
    Copyright (c) 2015 Arduino LLC. All rights reserved.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#pragma once

#include <Arduino.h>
#include "api/HardwareI2C.h"
#include <hardware/i2c.h>

// WIRE_HAS_END means Wire has end()
#define WIRE_HAS_END 1

#ifndef WIRE_BUFFER_SIZE
#define WIRE_BUFFER_SIZE 256
#endif

class TwoWire : public HardwareI2C {
public:
    TwoWire(i2c_inst_t *i2c, pin_size_t sda, pin_size_t scl);

    // Start as Master
    void begin() override;
    // Start as Slave
    void begin(uint8_t address) override;
    // Shut down the I2C interface
    void end() override;

    // Select IO pins to use.  Call before ::begin()
    bool setSDA(pin_size_t sda);
    bool setSCL(pin_size_t scl);

    void setClock(uint32_t freqHz) override;

    void beginTransmission(uint8_t) override;
    uint8_t endTransmission(bool stopBit) override;
    uint8_t endTransmission(void) override;

    size_t requestFrom(uint8_t address, size_t quantity, bool stopBit) override;
    size_t requestFrom(uint8_t address, size_t quantity) override;

    size_t write(uint8_t data) override;
    size_t write(const uint8_t * data, size_t quantity) override;

    virtual int available(void) override;
    virtual int read(void) override;
    virtual int peek(void) override;
    virtual void flush(void) override;
    void onReceive(void(*)(int));
    void onRequest(void(*)(void));

    inline size_t write(unsigned long n) {
        return write((uint8_t)n);
    }
    inline size_t write(long n) {
        return write((uint8_t)n);
    }
    inline size_t write(unsigned int n) {
        return write((uint8_t)n);
    }
    inline size_t write(int n) {
        return write((uint8_t)n);
    }
    using Print::write;

    // DMA/asynchronous transfers.  Do not combime with synchronous runs or bad stuff will happen
    // All buffers must be valid for entire DMA and not touched until `finished()` returns true.
    bool writeAsync(uint8_t address, const void *buffer, size_t bytes, bool sendStop);
    bool readAsync(uint8_t address, void *buffer, size_t bytes, bool sendStop);
    bool finishedAsync(); // Call to check if the async operations is completed and the buffer can be reused/read
    void abortAsync(); // Cancel an outstanding async I2C operation

    void setTimeout(uint32_t timeout = 25, bool reset_with_timeout = false);     // sets the maximum number of milliseconds to wait
    bool getTimeoutFlag(void);
    void clearTimeoutFlag(void);

    // IRQ callback
    void onIRQ();

private:
    i2c_inst_t *_i2c;
    pin_size_t _sda;
    pin_size_t _scl;
    int _clkHz;

    bool _running;
    bool _slave;
    uint8_t _addr;
    bool _txBegun;

    bool _timeoutFlag;
    bool _reset_with_timeout;
    void _handleTimeout(bool reset);

    uint8_t _buff[WIRE_BUFFER_SIZE];
    int _buffLen;
    int _buffOff;

    // Callback user functions
    void (*_onRequestCallback)(void);
    void (*_onReceiveCallback)(int);

    bool _slaveStartDet = false;

    // TWI clock frequency
    static const uint32_t TWI_CLOCK = 100000;

    // DMA
    bool _rxBegun = false;
    int _channelDMA;
    int _channelSendDMA;
    uint16_t *_dmaBuffer = nullptr;
    uint16_t *_dmaSendBuffer = nullptr;
    int _dmaBytes;
    uint8_t *_rxFinalBuffer;
};

extern TwoWire Wire;
extern TwoWire Wire1;
